function [ ] = GHGM0_map_setup( )
%
% Greenhouse Gas Module - function 0 (GHGM0)
%
% This file loads and formats the raw raster maps that are needed for the
% greenhouse gas module in addition to the maps that are used in the
% allocation and intensity modules. The vectorized maps are saved in excel
% and matlab-data formats.
%
% The function also loads the MAF sequestration tables and saves them in
% the matlab-data format.
%
% This function is not run during the execution of the intensity module
% (or LURNZ). It is run once to setup the maps that LURNZ takes as input.
% If and only if the raw / underlying data is updated should this script be
% rerun.
%

% Code by Simon Anastasiadis : 2012-03-09

%% Parameters

% Resolution
% This depends on the resolution of the underlying maps, currently only two
% resolutions are available: 25ha per pixel and 1ha per pixel.
resolution = 25;
% resolution = 1; % CURRENTLY NOT READY!

% output path
output_path = '..\Data for Modules\Data 4 Greenhouse Gas Module\';

%% Setup path

if resolution==25
    load_folder = 'Raw map data 25ha';
elseif resolution==1
    load_folder = 'Raw map data 1ha';
end

% shared load path
map_path = '..\Data for Modules\Data 4 Greenhouse Gas Module\';
% specific load path
map_path = [map_path, load_folder];

%% Load lookup tables

% load look up tables
lookup_tables.post89 = xlsread([output_path,'Sequestration tables.xls'],'post89','A2:K52');
lookup_tables.pre90  = xlsread([output_path,'Sequestration tables.xls'],'pre90','A2:K43');
% column 1 = age
% columns 2-10 = total pine sequestration by region
% column 11 = indigenous forest / scrub sequestration

% regions:
% Col  2 = Auckland = TA 1-15
% Col  3 = Waikato / Taupo = TA 16-21, 36
% Col  4 = Bay of Plenty = TA 22-27
% Col  5 = Gisborne = TA 28
% Col  6 = Hawkes Bay / Southern North Island = TA 29-35, 37-48
% Col  7 = Nelson / Marlborough = TA 51-54
% Col  8 = Canterbury / West oast = 55-66
% Col  9 = Otago = TA 68-71
% Col 10 = Southland = TA 72-75

%%  Save Matlab output

% remove unwanted files
clear('base_map', 'column_names', 'load_folder', 'msg', 'nrow', ...
      'file_name', 'map_path', 'ncol', 'sea_pixel');
%

% save output
save([output_path,'GHGM_input_data_',num2str(resolution),'ha'])

%% Inform User saving land use change is complete

msg = sprintf(' - - GHGM0 : setup data complete');
disp(msg)

end

% Subfunction : Construct GIS Header info from ASCI files

function GISheader = ConstructGISHeader(textdata, output_path, resolution )

% Map_GISHeader = [Longitude;Latitude;Xll;Yll;CellSize;No_Value]
% We construct the GISheader from map.textdata
% map.text data should be identical for all maps by construction

Longitude = sscanf(char(textdata(1)),'%*s %f');
Latitude  = sscanf(char(textdata(2)),'%*s %f');
Xll = sscanf(char(textdata(3)),'%*s %f');
Yll = sscanf(char(textdata(4)),'%*s %f');
CellSize  = sscanf(char(textdata(5)),'%*s %f');
No_Value  = sscanf(char(textdata(6)),'%*s %f');

% map.textdata contains the GISheader info
% char converts it from a cell array to a character string
% sscanf breaks the string into its component pieces

GISheader = [Longitude; Latitude; Xll; Yll; CellSize; No_Value];

%% Save GIS header information to excel file
% note that LUAM loads GISheader from the Matlab data file

% file and sheet name
file_name = ['GIS information - ',num2str(resolution),'ha.xls'];
sheet_name = 'Sheet1';

% header labels
GIS_labels = {'ncols'; 'nrows'; 'xllcorner'; 'yllcorner'; ...
              'cellsize'; 'NODATA_value'};

% write to Excel
xlswrite([output_path,file_name],GIS_labels,sheet_name,'A1');
xlswrite([output_path,file_name],GISheader ,sheet_name,'B1');

end
